/*:
 * @target MZ
 * @plugindesc Displays a transparent HUD showing the current day, date, time, HP, stamina, gold, and register when switch 0033 is ON.
 * @author Clay Davis
 * @help This plugin displays a transparent HUD showing the current day, date, time, HP, stamina, gold, and register when switch 0033 is ON.
 */

(() => {
    const CLOCK_SYSTEM_SWITCH_ID = 1; // ID of the switch to control the clock system
    const CURRENT_TIME_HUD_SWITCH_ID = 33; // ID of the switch to control the HUD display

    // Function to get the day of the week based on MP of Actor #4
    function getDayOfWeek(mp) {
        const days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
        return days[mp % 7];
    }

    // Function to get the current day of the month based on MP of Actor #4
    function getDayOfMonth(mp) {
        return (mp % 28) + 1;
    }

    // Function to get the current time based on MP of Actor #3
    function getCurrentTime(mp) {
        const totalMinutes = mp % 1440;
        const hours = Math.floor(totalMinutes / 60);
        const minutes = totalMinutes % 60;
        const isPM = hours >= 12;
        const displayHours = hours % 12 === 0 ? 12 : hours % 12;
        const period = isPM ? 'PM' : 'AM';
        return `${displayHours}:${minutes.toString().padStart(2, '0')} ${period}`;
    }

    // Function to get the current season based on MP of Actor #5
    function getSeason(mp) {
        if (mp >= 0 && mp <= 27) return 'Spring';
        if (mp >= 28 && mp <= 55) return 'Summer';
        if (mp >= 56 && mp <= 83) return 'Fall';
        if (mp >= 84 && mp <= 112) return 'Winter';
        return 'Unknown Season';
    }

    // Function to get the current year based on MP of Actor #6
    function getYear(mp) {
        return `Year ${mp + 1}`;
    }

    // Create a new sprite for the HUD
    class Sprite_HUD extends Sprite {
        initialize() {
            super.initialize();
            this.createBitmap();
            this.update();
        }

        createBitmap() {
            this.bitmap = new Bitmap(Graphics.width, Graphics.height);
            this.bitmap.fontFace = $gameSystem.mainFontFace(); // Default font
            this.bitmap.fontSize = 23;
            this.x = 20;
            this.y = 20;
        }

        drawOutlinedText(text, x, y, maxWidth, align = 'left') {
            const context = this.bitmap.context;
            context.font = `${this.bitmap.fontSize}px ${this.bitmap.fontFace}`;
            context.textAlign = align;

            // Draw the outline (black color)
            context.fillStyle = 'black';
            for (let offsetX = -2; offsetX <= 2; offsetX++) {
                for (let offsetY = -2; offsetY <= 2; offsetY++) {
                    if (offsetX !== 0 || offsetY !== 0) {
                        context.fillText(text, x + offsetX, y + offsetY + this.bitmap.fontSize);
                    }
                }
            }

            // Draw the main text (white color)
            context.fillStyle = 'white';
            context.fillText(text, x, y + this.bitmap.fontSize);
        }

        update() {
            this.bitmap.clear();
            if ($gameSwitches.value(CLOCK_SYSTEM_SWITCH_ID) && $gameSwitches.value(CURRENT_TIME_HUD_SWITCH_ID)) {
                const actorSlot4 = $gameActors.actor(4); // Day System
                const actorSlot3 = $gameActors.actor(3); // Time System
                const actorSlot5 = $gameActors.actor(5); // Season System
                const actorSlot6 = $gameActors.actor(6); // Year System
                const actorSlot1 = $gameActors.actor(1); // Stamina System
                const actorSlot90 = $gameActors.actor(90); // Register actor

                if (actorSlot4 && actorSlot3 && actorSlot5 && actorSlot6 && actorSlot1 && actorSlot90) {
                    const currentDay = getDayOfWeek(actorSlot4.mp);
                    const currentDayOfMonth = getDayOfMonth(actorSlot4.mp);
                    const currentTime = getCurrentTime(actorSlot3.mp);
                    const currentSeason = getSeason(actorSlot5.mp);
                    const currentYear = getYear(actorSlot6.mp);
                    
                    // Stamina (MP) and HP
                    const currentStamina = actorSlot1.mp;
                    const maxStamina = actorSlot1.mmp;
                    const currentHP = actorSlot1.hp;
                    const maxHP = actorSlot1.mhp;
                    
                    // Gold amount
                    const goldAmount = $gameParty.gold();
                    const registerAmount = actorSlot90.mp; // Register from Actor 90's MP

                    // Display information
                    const fullDate = `${currentDay}, ${currentDayOfMonth} ${currentSeason}, ${currentYear}`;
                    const hpAndStamina = `HP: ${currentHP}/${maxHP}, Stamina: ${currentStamina}/${maxStamina}`;
                    const goldInfo = `Gold: ${goldAmount}G, Register: ${registerAmount}G`;
                    const timeInfo = `${currentTime}`;

                    this.drawOutlinedText(fullDate, 0, 0, Graphics.width - 40, 'left');
                    this.drawOutlinedText(hpAndStamina, 0, 36, Graphics.width - 40, 'left');
                    this.drawOutlinedText(goldInfo, 0, 72, Graphics.width - 40, 'left');
                    this.drawOutlinedText(timeInfo, 0, 108, Graphics.width - 40, 'left');
                } else {
                    this.drawOutlinedText('One or more actor slots do not exist.', 0, 0, Graphics.width - 40, 'left');
                }
            }
        }
    }

    // Add the HUD sprite to the Scene_Map
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        this.createHUD();
    };

    Scene_Map.prototype.createHUD = function() {
        this._hudSprite = new Sprite_HUD();
        this.addChild(this._hudSprite);
    };

    // Update the HUD sprite each frame
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this._hudSprite) {
            this._hudSprite.update();
        }
    };
})();